; Copyright (c) 2004, Stubbe F. Hviid.  All rights reserved.
;	Unauthorized reproduction prohibited.
;+
; NAME:
;	FUNCTION p_set_struct_tag, struct, tag, value, tag2, value2
;
; PURPOSE:
;	Add or replace a single tag of an IDL structure variable
;
; CALLING SEQUENCE:
;	 FUNCTION p_set_struct_tag(struct, tag, value, [tag2], [value2])
;
; INPUTS:
;   struct:		The structure to be modified
;	tag:		The name of the tag to added/replaced
;	value:		The value to be inserted
;
; OPTIONAL INPUT:
;	tag2:		second tag to be inserted
;   value2:		second value to be inserted
;
; RETURNS:
;	The modified structure
;
; KEYWORD PARAMETERS:
;	NONE
;
; EXAMPLE:
;		str = {a: 0, b: 1, c: 2}
;		nstr = p_set_struct_tag(str, 'a', [0,1,2,3,4,5,6])   ; replace tag str.a
;		nstr = p_set_struct_tag(str, 'c', [0,1,2,3,4,5,6])   ; add new tag nstr.c
;
;		nstr = p_set_struct_tag(str, 'c', [0,1,2,3,4,5,6], 'd', 'Hello')   ; add new tags nstr.c and nstr.d
;
; MODIFICATION HISTORY:
; 	Written by:	Stubbe F. Hviid, 12/05-2004
;-
FUNCTION p_set_struct_tag, struct, tag, value, tag2, value2
	
	if n_elements(tag) eq 0 then return, -1
	if n_elements(value) eq 0 then return, -1
	; first handle case where input structure is empty
	if size(struct, /type) ne 8 then begin
		if n_elements(tag2) ne 0 AND n_elements(value2) ne 0 then begin
			return, create_struct(tag, value, tag2, value2)
		endif else begin
			return, create_struct(tag, value)
		endelse
	endif
	; make sure that the tags are uppercase
	tag = strupcase(tag)
	if n_elements(tag2) ne 0 then tag2 = strupcase(tag2)
	; extract the existing tag names from the input structure
	names = tag_names(struct)
	; find all tags in the existing structure not equal to the input tags
	if n_elements(tag2) ne 0 AND n_elements(value2) ne 0 then begin
		idx = where(names ne strupcase(tag) AND names ne strupcase(tag2))
		out = create_struct(tag, value, tag2, value2)
	endif else begin
		idx = where(names ne strupcase(tag))
		out = create_struct(tag, value)
	endelse
	; check if any existing tags needs to be added
	; no - then just return
	if idx[0] lt 0 then return, out
	; yes - then add them
	for i=0, n_elements(idx)-1 do begin
		out = create_struct(names[idx[i]], struct.(idx[i]), out)
	endfor
	return, out
END