; Copyright (c) 2004, Stubbe F. Hviid.  All rights reserved.
;	Unauthorized reproduction prohibited.
;+
; NAME:
;	FUNCTION p_format_pds_value, value, unit, LABEL=LABEL, _EXTRA=_EXTRA
;
; PURPOSE:
;	Format a value into a PDS formated text string
;
; CALLING SEQUENCE:
;	 Result = p_format_pds_value, value, unit, LABEL=LABEL
;
; INPUTS:
;	value: IDL scalar of 1D array of scalars. scalar includes strings
;
; OPTIONAL INPUT:
;   unit:   The physical unit of the item to be written
;
; RETURNS:
;	PDS formated value string
;
; KEYWORD PARAMETERS:
;	LABEL:		Force strings to be handled as labels
;   NO_QUOTES:  Write label strings without the single quotes around
;
; EXAMPLE:
;   print, p_format_pds_value(1.05, 's')					; returns	1.05 <s>
;   print, p_format_pds_value('HELLO', /LABEL)  			; returns   'HELLO'
;   print, p_format_pds_value('HELLO', /LABEL, /NO_QUOTES)  ; returns   HELLO
;   print, p_format_pds_value([1.05, 2.05], 's')					; returns	(1.05 <s>, 2.05 <s>)
;   print, p_format_pds_value(['Hello', 'World'])					; returns	('Hello', 'World')
;
; MODIFICATION HISTORY:
; 	Written by:	Stubbe F. Hviid, 11/8-2004
;-


FUNCTION p_format_pds_scalar, value, iunit, LABEL=LABEL, NO_QUOTES=NO_QUOTES, _EXTRA=_EXTRA

	if n_elements(iunit) ne 0 then unit = ' <' + iunit + ">" else unit = ''

	type = size(value, /TYPE)

	; scalars
	if (type gt 0 AND type le 5) OR (type ge 12 AND type le 15) then $
		return, strtrim(string(value, _EXTRA=_EXTRA), 2) + unit

	; complex
	if type eq 6 then begin
		print, 'COMPLEX NOT SUPPORTED'
		return, 'NULL'
	endif

	; strings
	if type eq 7 then begin
		if keyword_set(LABEL) then begin
			if keyword_set(NO_QUOTES) then begin
				return, value
			endif else begin
				return, "'" + value + "'"
			endelse
		endif else begin
			return, '"' + value + '"'
		endelse
	endif

	return, 'NULL'
END


FUNCTION p_format_pds_value, value, unit, _EXTRA=_EXTRA

	dim = size(value, /DIM)
	ndim = n_elements(dim)

	if ndim gt 1 then begin
		print, 'ONLY 0 or 1 dim object can be PDS formated
		return, 'NULL'
	endif

	count = n_elements(value)

	out = ''

	if count gt 1 then out = '('

	for i=0, count-1 do begin
		if i gt 0 then begin
			out = out + ', ' + p_format_pds_scalar(value[i], unit, _EXTRA=_EXTRA)
		endif else begin
			out = out + p_format_pds_scalar(value[i], unit, _EXTRA=_EXTRA)
		endelse
	endfor

	if count gt 1 then out = out + ')'

	return, out

END