; Copyright (c) 2004, Stubbe F. Hviid.  All rights reserved.
;   Unauthorized reproduction prohibited.
;+
; NAME:
;   FUNCTION p_read_image_header, header
;
; PURPOSE:
;   Extract an image header object from a PDS data source
;
; CALLING SEQUENCE:
;   Result = p_read_image_header(header)
;
; INPUTS:
;   header:   		Valid PDS header structure
;
; KEYWORDS:
;	VERBOSE:		If set progress messages will be printed to the console
;	OK				Will be set to 1 on success and 0 on error
;
; RETURNS:
;   on success - A structure of the form:
;                 structure = {type: <image header type
;                              format: <either ASCII or BINARY>    - optional member default: ASCII
;                              header: <byte (BINARY) or string (ASCII) array with header data>
;                              description: <Extrenal file reference descriping the header> - optional input
;                             }
;	  -1 on error
;
; EXAMPLE:
;   p_rhead, pickfile(), h
;   ih = p_read_image_header(h)
;
; MODIFICATION HISTORY:
;   Written by:  Stubbe F. Hviid, 24/2-2008
;-

FUNCTION p_read_image_header, header, VERBOSE=VERBOSE, OK=OK

  offset = p_value(header, '^IMAGE_HEADER')

  if offset eq '' then return, -1

  type = p_value(header, 'IMAGE_HEADER.HEADER_TYPE')
  bytes = p_value(header, 'IMAGE_HEADER.BYTES')
  format = p_value(header, 'IMAGE_HEADER.INTERCHANGE_FORMAT')
  description = p_value(header, 'IMAGE_HEADER.^DESCRIPTION')

  record_bytes = p_value(header, 'RECORD_BYTES')

  offset = (offset - 1) * record_bytes

	on_ioerror, handle_error

	openr, unit, header.filename, /GET_LUN

	;point_lun, unit, img_offset
	if offset gt 0 AND bytes gt 0 then begin
		if keyword_set(VERBOSE) then print, 'Image Header data found at offset: ' + strtrim(offset, 2)

		point_lun, unit, offset

		data = bytarr(bytes)
		readu, unit, data

		close, unit
    free_lun, unit
	endif else return, -1

	if format eq 'ASCII' then begin
	  data = string(data)
	  sep = string([byte('0D'X), byte('0A'X)])
	  if strpos(data, sep) ge 0 then begin
	    data = strsplit(data, sep, /EXTRACT)
	  endif
	endif

  ; mark ok
  OK = 1

  ; return image header structure
  if description ne '' then begin
    return, {type: type, format: format, header: data, description: description}
  endif

  return, {type: type, format: format, header: data}

  ; handle error
	handle_error:
	print, 'ERROR: Error reading image header'
	OK = 0
	return, -1


END