; Copyright (c) 2004, Stubbe F. Hviid.  All rights reserved.
;   Unauthorized reproduction prohibited.
;+
; NAME:
;   FUNCTION p_read_image_object, header, object_name
;
; PURPOSE:
;   Extract an image object from a PDS data source
;
; CALLING SEQUENCE:
;   Result = p_read_image_object(header, object_name)
;
; INPUTS:
;   header:   		Valid PDS header structure
;   object_name:	The name of the image object to extract
;					(If not specified the 'IMAGE' is assumed)
;
; KEYWORDS:
;	VERBOSE:		If set progress messages will be printed to the console
;	OK				Will be set to 1 on success and 0 on error
;
; RETURNS:
;   The extract image matrix on success
;	-1 on error
;
; EXAMPLE:
;   p_rhead, pickfile(), h
;   tvscl, p_read_image_object(h, 'IMAGE')
;
; MODIFICATION HISTORY:
;   Written by:  Stubbe F. Hviid, 31/08-2006
;-

FUNCTION p_read_image_object, header, object_name, VERBOSE=VERBOSE, OK=OK

	if n_elements(object_name) eq 0 then object_name = 'IMAGE'

	if keyword_set(VERBOSE) then print, 'Reading data object: ' + object_name + ' from file: ' + header.filename

	file = header.filename
	data_file = header.filename

	pimage = strtrim(p_value(header, '^' + object_name, /AS_STRING), 2)
	if strmid(pimage, 0, 1) eq '(' then begin
		; parse external data reference
		data_file = strmid(pimage, 2)
		p = strpos(data_file, '"')

		remain = strmid(data_file, p)
		data_file = strmid(data_file, 0, p)

		p = strpos(remain, ',')
		data_pointer = long(strmid(remain, p+1))
		img_offset = long(p_value(header, 'RECORD_BYTES')) * (data_pointer - 1)	> 0

		lp = p_parse_file_path(file)
		dp = p_parse_file_path(data_file)

		if dp.dir eq '.' and lp.dir ne '.' then data_file = lp.dir + '/' + data_file


	endif else begin
		if strpos(pimage, '<BYTES>') gt 0 then begin
			img_offset = long(pimage)
		endif else begin
			data_pointer = long(pimage)
			img_offset = long(p_value(header, 'RECORD_BYTES')) * (data_pointer - 1)
		endelse
	endelse


	w = long(p_value(header, object_name + '.LINE_SAMPLES'))
	h = long(p_value(header, object_name + '.LINES'))
	d = p_value(header, object_name + '.BANDS')
	if d eq '' then d = 1 else d = long(d)
	bd = long(p_value(header, object_name + '.SAMPLE_BITS'))

	if w eq 0 OR h eq 0 then begin
		print, 'ERROR: no image data found'
		OK = 0
		return, -1
	endif


	on_ioerror, handle_error

	openr, unit, data_file, /GET_LUN

	;point_lun, unit, img_offset
	if img_offset gt 0 then begin
		if keyword_set(VERBOSE) then print, 'Image data found at offset: ' + strtrim(img_offset, 2)
		tmp = bytarr(img_offset)
		readu, unit, tmp
	endif

	; ----------------------------------------------------------------------------
	; handle special case of VICAR images with binary header and/or binary prefix
	; (This is a rotten worksaround :-(  )
	if header.filetype eq 'VICAR' AND object_name eq 'IMAGE' then begin
		if fix(p_value(header, 'NLB')) gt 0 OR fix(p_value(header, 'NBB')) gt 0 then begin
			p_rvicar_img_with_bh, file, tmp, image, OK=OK, _EXTRA=_EXTRA
			return, image
		endif
	endif

	; ----------------------------------------------------------------------------
	; Perform std read operations
	type = p_value(header, object_name + '.SAMPLE_TYPE')

	if type eq 'LSB_UNSIGNED_INTEGER' then begin
		if keyword_set(VERBOSE) then print, 'Datatype: PC uint ' + strtrim(bd/8, 2) + ' byte'
		if bd eq 8 then image = bytarr(w,h,d)
		if bd eq 16 then image = uintarr(w,h,d)
		if bd eq 32 then image = ulonarr(w,h,d)
		readu, unit, image
		if bd gt 8 then byteorder, image, /SWAP_IF_BIG_ENDIAN

		if bd eq 8 then begin
			image = fix(image) + 127
		endif
	endif

	if type eq 'MSB_UNSIGNED_INTEGER' OR type eq 'UNSIGNED_INTEGER' then begin
		if keyword_set(VERBOSE) then print, 'Datatype: Motorola uint ' + strtrim(bd/8, 2) + ' byte'
		if bd eq 8 then image = bytarr(w,h,d)
		if bd eq 16 then image = uintarr(w,h,d)
		if bd eq 32 then image = ulonarr(w,h,d)
		readu, unit, image
		if bd gt 8 then byteorder, image, /SWAP_IF_LITTLE_ENDIAN

;		if bd eq 8 then begin
;			image = fix(image) + 127
;		endif
	endif

	if type eq 'LSB_INTEGER' then begin
		if keyword_set(VERBOSE) then print, 'Datatype: PC int ' + strtrim(bd/8, 2) + ' byte'
		if bd eq 8 then image = bytarr(w,h,d)
		if bd eq 16 then image = intarr(w,h,d)
		if bd eq 32 then image = lonarr(w,h,d)
		readu, unit, image
		if bd eq 16 then byteorder, image, /SWAP_IF_BIG_ENDIAN
		if bd eq 32 then byteorder, image, /SWAP_IF_BIG_ENDIAN, /LSWAP
	endif

	if type eq 'MSB_INTEGER' OR type eq 'INTEGER' then begin
		if keyword_set(VERBOSE) then print, 'Datatype: Motorola int ' + strtrim(bd/8, 2) + ' byte'
		if bd eq 8 then image = bytarr(w,h,d)
		if bd eq 16 then image = intarr(w,h,d)
		if bd eq 32 then image = lonarr(w,h,d)
		readu, unit, image
		if bd eq 16 then byteorder, image, /SWAP_IF_LITTLE_ENDIAN
		if bd eq 32 then byteorder, image, /SWAP_IF_LITTLE_ENDIAN, /LSWAP
	endif

	if type eq 'PC_REAL' then begin
		if keyword_set(VERBOSE) then print, 'Datatype: PC real ' + strtrim(bd/8, 2) + ' byte'

		if bd eq 32 then image = fltarr(w,h,d)
		if bd eq 64 then image = dblarr(w,h,d)
		readu, unit, image

		if bd eq 64 then begin
			byteorder, image, /SWAP_IF_BIG_ENDIAN, /L64SWAP
		endif else begin
			byteorder, image, /SWAP_IF_BIG_ENDIAN, /LSWAP
		endelse
	endif

	if type eq 'IEEE_REAL' then begin
		if keyword_set(VERBOSE) then print, 'Datatype: Motorola real ' + strtrim(bd/8, 2) + ' byte'

		if bd eq 32 then image = fltarr(w,h,d)
		if bd eq 64 then image = dblarr(w,h,d)
		readu, unit, image

		if bd eq 64 then begin
			byteorder, image, /SWAP_IF_LITTLE_ENDIAN, /L64SWAP
		endif else begin
			byteorder, image, /SWAP_IF_LITTLE_ENDIAN, /LSWAP
		endelse
	endif

	close, unit
	free_lun, unit

	if n_elements(image) ne 0 then image = reform(image)

	offset = p_value(header, object_name + '.OFFSET', FOUND=found)
	if found ne 0 then begin
		if keyword_set(VERBOSE) then print, 'Found OFFSET(' + strtrim(offset, 2) + ') specifier in image object - adding offset to image data'
		image = image + offset
	endif

	OK = 1
	return, image

	handle_error:
	print, 'ERROR: reading image data'
	OK = 0
	return, -1


END