; Copyright (c) 2004, Stubbe F. Hviid.  All rights reserved.
;	Unauthorized reproduction prohibited.
;+
; NAME:
;	FUNCTION p_open_stripeset, file, stripe_height
;
; PURPOSE:
;	Function for handling huge image data files. A stripeset is an interface for reading the image data in segments from the disk
;   small enough that the image data will fit in memory.
; 	The function supports PDS and VICAR images
;
; CALLING SEQUENCE:
;	 result = p_open_stripeset(file, stripe_height)
;
; INPUTS:
;   file:   		The file to read from
;   stripe_height:  Number of rows of image data to be read per cycle
;
; RETURNS:
;	Result is a structure defining the stripeset
;
; KEYWORD PARAMETERS:
;	None
;
; EXAMPLE:
;	st = p_open_stripeset(pickfile(), 2000)		; open stripe set using 2000 liner per read cycle
;   ok=1
;   while ok do begin
; 		image = p_read_stripe(st, OK=ok)	; read a stripe segment OK returns 0 on read failure (end of image)
; 		if ok ne 0 then tvscl, image
;   endwhile
;   p_stripe_close, st
;
; MODIFICATION HISTORY:
; 	Written by:	Stubbe F. Hviid, 12/05-2004
;-

FUNCTION p_open_stripeset, file, stripe_height, image_object

	p_rhead, file, header, OK=OK

	if OK eq 0 then return, -1

	if n_elements(image_object) eq 0 then image_object = 'IMAGE'

	pimage = strtrim(p_value(header, '^' + image_object, /AS_STRING), 2)
	if strmid(pimage, 0, 1) eq '(' then begin
		; parse external data reference
		data_file = strmid(pimage, 2)
		p = strpos(data_file, '"')

		remain = strmid(data_file, p)
		data_file = strmid(data_file, 0, p)

		p = strpos(remain, ',')
		data_pointer = long(strmid(remain, p+1))
		img_offset = long(p_value(header, 'RECORD_BYTES')) * (data_pointer - 1)	> 0

		lp = p_parse_file_path(file)
		dp = p_parse_file_path(data_file)

		if dp.dir eq '.' and lp.dir ne '.' then data_file = lp.dir + '/' + data_file


	endif else begin
		if strpos(pimage, '<BYTES>') gt 0 then begin
			img_offset = long(pimage)
		endif else begin
			data_pointer = long(pimage)
			img_offset = long(p_value(header, 'RECORD_BYTES')) * (data_pointer - 1)
		endelse
	endelse

	w = long(p_value(header, 'IMAGE.LINE_SAMPLES'))
	h = long(p_value(header, 'IMAGE.LINES'))
	d = p_value(header, 'IMAGE.BANDS')
	if d eq '' then d = 1 else d = long(d)
	bd = long(p_value(header, 'IMAGE.SAMPLE_BITS'))

	if w eq 0 OR h eq 0 then begin
		print, 'no image data found'
		return, -1
	endif

	if d ne 1 then begin
		print, 'Image with multible planes not supported in stripesets'
		return, -1
	endif

	on_ioerror, handle_error

	; open file for reading
	openr, unit, file, /GET_LUN

	; strip header away
	binhdr = bytarr(img_offset)
	readu, unit, binhdr

	; determine type
	type = p_value(header, 'IMAGE.SAMPLE_TYPE')

	if type eq 'LSB_UNSIGNED_INTEGER' then begin
		if bd eq 8 then image = bytarr(w,stripe_height)
		if bd eq 16 then image = uintarr(w,stripe_height)
		if bd eq 32 then image = ulonarr(w,stripe_height)
		endian = 'little'
	endif

	if type eq 'MSB_UNSIGNED_INTEGER' then begin
		if bd eq 8 then image = bytarr(w,stripe_height)
		if bd eq 16 then image = uintarr(w,stripe_height)
		if bd eq 32 then image = ulonarr(w,stripe_height)
		endian = 'big'
	endif

	if type eq 'LSB_INTEGER' then begin
		if bd eq 8 then image = bytarr(w,stripe_height)
		if bd eq 16 then image = intarr(w,stripe_height)
		if bd eq 32 then image = lonarr(w,stripe_height)
		endian = 'little'
	endif

	if type eq 'MSB_INTEGER' then begin
		if bd eq 8 then image = bytarr(w,stripe_height)
		if bd eq 16 then image = intarr(w,stripe_height)
		if bd eq 32 then image = lonarr(w,stripe_height)
		endian = 'big'
	endif

	if type eq 'PC_REAL' then begin
		if bd eq 32 then image = fltarr(w,stripe_height)
		if bd eq 64 then image = dblarr(w,stripe_height)
		endian = 'little'
	endif

	if type eq 'IEEE_REAL' then begin
		if bd eq 32 then image = fltarr(w,stripe_height)
		if bd eq 64 then image = dblarr(w,stripe_height)
		endian = 'big'
	endif

	return, {type: 'STRIPEHEAD', header: header, unit: unit, next_line: 0, stripe_height: stripe_height, endian: endian, datatype: size(image, /TYPE), w: w, h: h}

	handle_error:
	print, 'Error reading file'
	return, -1
END
